/*  An Inkview application for downloading files and saving them locally.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <libgen.h>
#include <stdarg.h>
#include <inkview.h>

#define PBDLCFGFILE  CONFIGPATH "/pbdownload.cfg"

static int ret = 0;
static char *url, *lasturl;
static char *outfile, *lastfile;
static char *postdata = NULL;
static int timeout = 60;
static int sid;
#define bufsize 1024
static char progmsg[bufsize];
static char kbuf[bufsize];
static iconfig *pbdlcfg = NULL;

iconfigedit pbdlce[] = {
	{ CFG_INFO, NULL, "pbdownload previous entries", NULL, NULL, NULL, NULL, NULL },
	{ CFG_TEXT, NULL, "Last URL", NULL, "last_url", NULL, NULL, NULL },
	{ CFG_TEXT, NULL, "Last File", NULL, "last_file", NULL, NULL, NULL },
	{ 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};

void output(const char *fmt, ...) {
	FILE *fp;

	fp = fopen("/tmp/dbg", "a");
	va_list args;
	va_start(args, fmt);
	vfprintf(fp, fmt, args);
	va_end(args);
	fclose(fp);
}

void read_config() {
	pbdlcfg = OpenConfig(PBDLCFGFILE, pbdlce);
	lasturl = ReadString(pbdlcfg, "last_url", "http://");
	lastfile = ReadString(pbdlcfg, "last_file", "/mnt/ext1/");
}

void save_config() {
	if (url && outfile) {
		WriteString(pbdlcfg, "last_url", url);
		WriteString(pbdlcfg, "last_file", outfile);
		SaveConfig(pbdlcfg);
	}
	CloseConfig(pbdlcfg);
}

void dlg_handler(int button) {
	ret = button;
	CloseApp();
}

void update_progress(void) {
	iv_sessioninfo *si;
	char errmsg[bufsize];

	si = GetSessionInfo(sid);

	if (si->length <= 0) {
		CloseProgressbar();
		sprintf(errmsg, "Couldn't connect to server at url: ");
		strcat(errmsg, url);
		Dialog(ICON_ERROR, "pbdownload", errmsg, GetLangText("@Close"), NULL, dlg_handler);
	} else {
		UpdateProgressbar(progmsg, 100*si->progress/si->length);

		if (si->progress >= si->length) {
			CloseSession(sid);
			CloseProgressbar();
			if (si->response >= 400) {
				sprintf(errmsg, "Server returned error response: %d\n", (int)si->response);
				Dialog(ICON_ERROR, "pbdownload", errmsg, GetLangText("@Close"), NULL, dlg_handler);
			} else
				CloseApp();
		} else
			SetHardTimer("pbdl_timer", update_progress, 1000);
	}
}

void start_download() {
	snprintf(progmsg, bufsize, "%s -> %s:\n", url, outfile);

	sid = NewSession();
	ret = DownloadTo(sid, url, postdata, outfile, timeout);
	OpenProgressbar(ICON_INFORMATION, "pbdownload", progmsg, 0, dlg_handler);
	SetHardTimer("pbdl_timer", update_progress, 1000);
}

void path_handler(char *path) {
	int ret;
	struct stat sb;
	char temp[bufsize], *dirc, *basec, *dir;

	if (path && *path) {
		ret = stat(path, &sb);
		if (ret == 0) {
			/* We have something that already exists, and that we can access. */
			if (S_ISDIR(sb.st_mode)) {
				/* A directory name was given, so append the filename from the URL. */
				strcpy(temp, path);
				if (temp[strlen(temp)-1] != '/')
					strcat(temp, "/");
				basec = strdup(url);
				strcat(temp, basename(basec));
				free(basec);
				outfile = strdup(temp);
			} else if (S_ISREG(sb.st_mode)) {
				/* We have an existing file, so overwrite it. */
				outfile = strdup(path);
			} else {
				/* This is not a regular file or directory. */
				sprintf(temp, "Output file %s is not a regular file or directory.  Giving up!", path);
				Dialog(ICON_ERROR, "pbdownload", temp, GetLangText("@Close"), NULL, dlg_handler);
			}
		} else {
			/* We have something that doesn't exist yet or is inaccessible. */
			if (errno != ENOENT) {
				/* Something is wrong besides the file not existing yet. */
				sprintf(temp, "Bad output name %s: %s", path, strerror(errno));
				Dialog(ICON_ERROR, "pbdownload", temp, GetLangText("@Close"), NULL, dlg_handler);
			} else if (path[strlen(path)-1] == '/') {
				/* A non-existent target directory was given. */
				sprintf(temp, "Output directory does not exist.  Giving up!");
				Dialog(ICON_ERROR, "pbdownload", temp, GetLangText("@Close"), NULL, dlg_handler);
			} else {
				/* Check to see if the directory path exists. */
				dirc = strdup(path);
				dir = dirname(dirc);
				if (stat(dir, &sb) == 0)
					outfile = strdup(path);
				else {
					sprintf(temp, "Output directory %s/ does not exist.  Giving up!", dir);
					Dialog(ICON_ERROR, "pbdownload", temp, GetLangText("@Close"), NULL, dlg_handler);
				}
				free(dirc);
			}
		}
		if (outfile)
			start_download();
	} else {
		ret = -1;
		CloseApp();
	}
}

void dir_handler(char *dir) {
	char *basec;
	if (dir && *dir) {
		if (kbuf[strlen(kbuf)-1] != '/')
			strcat(kbuf, "/");
		if (strcmp(url, lasturl) == 0)
			basec = strdup(lastfile);
		else
			basec = strdup(url);
		strcat(kbuf, basename(basec));
		free(basec);
		OpenKeyboard("Enter output file name (or directory):", kbuf, bufsize, KBD_URL, path_handler);
	} else {
		ret = -1;
		CloseApp();
	}
}

void get_path() {
	char *dirc;

	if (!outfile) {
		dirc = strdup(lastfile);
		sprintf(kbuf, "%s", dirname(dirc));
		free(dirc);
		OpenDirectorySelector("Choose directory for output file:", kbuf, bufsize, dir_handler);
	} else
		start_download();
}

void url_handler(char *_url) {
	if (_url && *_url) {
		url = strdup(_url);
		get_path();
	} else {
		ret = -1;
		CloseApp();
	}
}

void get_url() {
	if (!url) {
		sprintf(kbuf, "%s", lasturl);
		OpenKeyboard("Enter URL:", kbuf, bufsize, KBD_URL, url_handler);
	} else 
		get_path();
}

int main_handler(int type, int par1, int par2) {

	if (type == EVT_INIT) {
		/* Start the download process. */
		read_config();
		get_url();
	}

	if (type == EVT_KEYREPEAT) {
		switch (par1) {
			case KEY_BACK:
				CloseApp();
				break;
			default:
				break;
		}
	}

	if (type == EVT_EXIT) {
		if (ret)
			iv_unlink(outfile);
		save_config();
		free(url);
		free(outfile);
		free(postdata);
	}

	return 0;
}

int main(int argc, char **argv)
{
	int opt;

	while ((opt = getopt(argc, argv, "f:u:t:p:h")) != -1) {
		switch (opt) {
			case 'f':
				outfile = strdup(optarg);
				break;
			case 'u':
				url = strdup(optarg);
				break;
			case 't':
				timeout = atoi(optarg);
				break;
			case 'p':
				postdata = strdup(optarg);
				break;
			default:
				printf("Usage: pbdownload -u <URL> -f <output-file-path> -t <timeout seconds> -p <post-data>\n");
				break;
		}
	}

	InkViewMain(main_handler);
	return ret;
}

