/*
 * bmp2ibitmap() converts a BMP file to an ibitmap structure.
 * This code was hacked out from bmp2tiff.c in tiff-3.9.2,
 * and the license in that file is reproduced below.
 */

/* $Id: bmp2tiff.c,v 1.20 2006/03/23 14:54:02 dron Exp $
 *
 * Project:  libtiff tools
 * Purpose:  Convert Windows BMP files in TIFF.
 * Author:   Andrey Kiselev, dron@ak4719.spb.edu
 *
 ******************************************************************************
 * Copyright (c) 2004, Andrey Kiselev <dron@ak4719.spb.edu>
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */

#include "tif_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#if HAVE_FCNTL_H
# include <fcntl.h>
#endif

#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#if HAVE_IO_H
# include <io.h>
#endif

#include "tiffio.h"
#include "img2ibitmap.h"

#ifndef O_BINARY
# define O_BINARY 0
#endif

enum BMPType
{
    BMPT_WIN4,      /* BMP used in Windows 3.0/NT 3.51/95 */
    BMPT_WIN5,      /* BMP used in Windows NT 4.0/98/Me/2000/XP */
    BMPT_OS21,      /* BMP used in OS/2 PM 1.x */
    BMPT_OS22       /* BMP used in OS/2 PM 2.x */
};

/*
 * Bitmap file consists of a BMPFileHeader structure followed by a
 * BMPInfoHeader structure. An array of BMPColorEntry structures (also called
 * a colour table) follows the bitmap information header structure. The colour
 * table is followed by a second array of indexes into the colour table (the
 * actual bitmap data). Data may be comressed, for 4-bpp and 8-bpp used RLE
 * compression.
 *
 * +---------------------+
 * | BMPFileHeader       |
 * +---------------------+
 * | BMPInfoHeader       |
 * +---------------------+
 * | BMPColorEntry array |
 * +---------------------+
 * | Colour-index array  |
 * +---------------------+
 *
 * All numbers stored in Intel order with least significant byte first.
 */

enum BMPComprMethod
{
    BMPC_RGB = 0L,          /* Uncompressed */
    BMPC_RLE8 = 1L,         /* RLE for 8 bpp images */
    BMPC_RLE4 = 2L,         /* RLE for 4 bpp images */
    BMPC_BITFIELDS = 3L,    /* Bitmap is not compressed and the colour table
			     * consists of three DWORD color masks that specify
			     * the red, green, and blue components of each
			     * pixel. This is valid when used with
			     * 16- and 32-bpp bitmaps. */
    BMPC_JPEG = 4L,         /* Indicates that the image is a JPEG image. */
    BMPC_PNG = 5L           /* Indicates that the image is a PNG image. */
};

enum BMPLCSType                 /* Type of logical color space. */
{
    BMPLT_CALIBRATED_RGB = 0,	/* This value indicates that endpoints and
				 * gamma values are given in the appropriate
				 * fields. */
    BMPLT_DEVICE_RGB = 1,
    BMPLT_DEVICE_CMYK = 2
};

typedef struct
{
    int32   iCIEX;
    int32   iCIEY;
    int32   iCIEZ;
} BMPCIEXYZ;

typedef struct                  /* This structure contains the x, y, and z */
{				/* coordinates of the three colors that */
				/* correspond */
    BMPCIEXYZ   iCIERed;        /* to the red, green, and blue endpoints for */
    BMPCIEXYZ   iCIEGreen;      /* a specified logical color space. */
    BMPCIEXYZ	iCIEBlue;
} BMPCIEXYZTriple;

typedef struct
{
    char	bType[2];       /* Signature "BM" */
    uint32	iSize;          /* Size in bytes of the bitmap file. Should
				 * always be ignored while reading because
				 * of error in Windows 3.0 SDK's description
				 * of this field */
    uint16	iReserved1;     /* Reserved, set as 0 */
    uint16	iReserved2;     /* Reserved, set as 0 */
    uint32	iOffBits;       /* Offset of the image from file start in bytes */
} BMPFileHeader;

/* File header size in bytes: */
const int       BFH_SIZE = 14;

typedef struct
{
    uint32	iSize;          /* Size of BMPInfoHeader structure in bytes.
				 * Should be used to determine start of the
				 * colour table */
    int32	iWidth;         /* Image width */
    int32	iHeight;        /* Image height. If positive, image has bottom
				 * left origin, if negative --- top left. */
    int16	iPlanes;        /* Number of image planes (must be set to 1) */
    int16	iBitCount;      /* Number of bits per pixel (1, 4, 8, 16, 24
				 * or 32). If 0 then the number of bits per
				 * pixel is specified or is implied by the
				 * JPEG or PNG format. */
    uint32	iCompression;	/* Compression method */
    uint32	iSizeImage;     /* Size of uncomressed image in bytes. May
				 * be 0 for BMPC_RGB bitmaps. If iCompression
				 * is BI_JPEG or BI_PNG, iSizeImage indicates
				 * the size of the JPEG or PNG image buffer. */
    int32	iXPelsPerMeter; /* X resolution, pixels per meter (0 if not used) */
    int32	iYPelsPerMeter; /* Y resolution, pixels per meter (0 if not used) */
    uint32	iClrUsed;       /* Size of colour table. If 0, iBitCount should
				 * be used to calculate this value
				 * (1<<iBitCount). This value should be
				 * unsigned for proper shifting. */
    int32	iClrImportant;  /* Number of important colours. If 0, all
				 * colours are required */

    /*
     * Fields above should be used for bitmaps, compatible with Windows NT 3.51
     * and earlier. Windows 98/Me, Windows 2000/XP introduces additional fields:
     */

    int32	iRedMask;       /* Colour mask that specifies the red component
				 * of each pixel, valid only if iCompression
				 * is set to BI_BITFIELDS. */
    int32	iGreenMask;     /* The same for green component */
    int32	iBlueMask;      /* The same for blue component */
    int32	iAlphaMask;     /* Colour mask that specifies the alpha
				 * component of each pixel. */
    uint32	iCSType;        /* Colour space of the DIB. */
    BMPCIEXYZTriple sEndpoints; /* This member is ignored unless the iCSType
				 * member specifies BMPLT_CALIBRATED_RGB. */
    int32	iGammaRed;      /* Toned response curve for red. This member
				 * is ignored unless color values are
				 * calibrated RGB values and iCSType is set to
				 * BMPLT_CALIBRATED_RGB. Specified
				 * in 16^16 format. */
    int32	iGammaGreen;    /* Toned response curve for green. */
    int32	iGammaBlue;     /* Toned response curve for blue. */
} BMPInfoHeader;

/*
 * Info header size in bytes:
 */
const unsigned int  BIH_WIN4SIZE = 40; /* for BMPT_WIN4 */
const unsigned int  BIH_WIN5SIZE = 57; /* for BMPT_WIN5 */
const unsigned int  BIH_OS21SIZE = 12; /* for BMPT_OS21 */
const unsigned int  BIH_OS22SIZE = 64; /* for BMPT_OS22 */

/*
 * We will use plain byte array instead of this structure, but declaration
 * provided for reference
 */
typedef struct
{
    char       bBlue;
    char       bGreen;
    char       bRed;
    char       bReserved;      /* Must be 0 */
} BMPColorEntry;


ibitmap* bmp2ibitmap(const char* infilename)
{
	uint32	width, length;
	uint16	nbands = 1;		/* number of bands in input image */
	uint16	depth = 8;		/* bits per pixel in input image */
	uint16	photometric = PHOTOMETRIC_MINISBLACK;
	int	fd = 0;
	struct stat instat;

	BMPFileHeader file_hdr;
        BMPInfoHeader info_hdr;
        int     bmp_type;
        uint32  clr_tbl_size, n_clr_elems = 3;
        unsigned char *clr_tbl = NULL;
	uint32	row, clr, i, j;
	unsigned char r, g, b, a;
	ibitmap *bit = NULL;

	fd = open(infilename, O_RDONLY|O_BINARY, 0);
	if (fd < 0) {
		TIFFError(infilename, "Cannot open input file");
		return NULL;
	}

	read(fd, file_hdr.bType, 2);
	if(file_hdr.bType[0] != 'B' || file_hdr.bType[1] != 'M') {
		TIFFError(infilename, "File is not BMP");
		goto bad;
	}

/* -------------------------------------------------------------------- */
/*      Read the BMPFileHeader. We need iOffBits value only             */
/* -------------------------------------------------------------------- */
	lseek(fd, 10, SEEK_SET);
	read(fd, &file_hdr.iOffBits, 4);
#ifdef WORDS_BIGENDIAN
	TIFFSwabLong(&file_hdr.iOffBits);
#endif
	fstat(fd, &instat);
	file_hdr.iSize = instat.st_size;

/* -------------------------------------------------------------------- */
/*      Read the BMPInfoHeader.                                         */
/* -------------------------------------------------------------------- */

	lseek(fd, BFH_SIZE, SEEK_SET);
	read(fd, &info_hdr.iSize, 4);
#ifdef WORDS_BIGENDIAN
	TIFFSwabLong(&info_hdr.iSize);
#endif

	if (info_hdr.iSize == BIH_WIN4SIZE)
		bmp_type = BMPT_WIN4;
	else if (info_hdr.iSize == BIH_OS21SIZE)
		bmp_type = BMPT_OS21;
	else if (info_hdr.iSize == BIH_OS22SIZE
		 || info_hdr.iSize == 16)
		bmp_type = BMPT_OS22;
	else
		bmp_type = BMPT_WIN5;

	if (bmp_type == BMPT_WIN4
	    || bmp_type == BMPT_WIN5
	    || bmp_type == BMPT_OS22) {
		read(fd, &info_hdr.iWidth, 4);
		read(fd, &info_hdr.iHeight, 4);
		read(fd, &info_hdr.iPlanes, 2);
		read(fd, &info_hdr.iBitCount, 2);
		read(fd, &info_hdr.iCompression, 4);
		read(fd, &info_hdr.iSizeImage, 4);
		read(fd, &info_hdr.iXPelsPerMeter, 4);
		read(fd, &info_hdr.iYPelsPerMeter, 4);
		read(fd, &info_hdr.iClrUsed, 4);
		read(fd, &info_hdr.iClrImportant, 4);
#ifdef WORDS_BIGENDIAN
		TIFFSwabLong((uint32*) &info_hdr.iWidth);
		TIFFSwabLong((uint32*) &info_hdr.iHeight);
		TIFFSwabShort((uint16*) &info_hdr.iPlanes);
		TIFFSwabShort((uint16*) &info_hdr.iBitCount);
		TIFFSwabLong((uint32*) &info_hdr.iCompression);
		TIFFSwabLong((uint32*) &info_hdr.iSizeImage);
		TIFFSwabLong((uint32*) &info_hdr.iXPelsPerMeter);
		TIFFSwabLong((uint32*) &info_hdr.iYPelsPerMeter);
		TIFFSwabLong((uint32*) &info_hdr.iClrUsed);
		TIFFSwabLong((uint32*) &info_hdr.iClrImportant);
#endif
		n_clr_elems = 4;
	}

	if (bmp_type == BMPT_OS22) {
		/* 
		 * FIXME: different info in different documents
		 * regarding this!
		 */
		 n_clr_elems = 3;
	}

	if (bmp_type == BMPT_OS21) {
		int16  iShort;

		read(fd, &iShort, 2);
#ifdef WORDS_BIGENDIAN
		TIFFSwabShort((uint16*) &iShort);
#endif
		info_hdr.iWidth = iShort;
		read(fd, &iShort, 2);
#ifdef WORDS_BIGENDIAN
		TIFFSwabShort((uint16*) &iShort);
#endif
		info_hdr.iHeight = iShort;
		read(fd, &iShort, 2);
#ifdef WORDS_BIGENDIAN
		TIFFSwabShort((uint16*) &iShort);
#endif
		info_hdr.iPlanes = iShort;
		read(fd, &iShort, 2);
#ifdef WORDS_BIGENDIAN
		TIFFSwabShort((uint16*) &iShort);
#endif
		info_hdr.iBitCount = iShort;
		info_hdr.iCompression = BMPC_RGB;
		n_clr_elems = 3;
	}

	if (info_hdr.iBitCount != 1  && info_hdr.iBitCount != 4  &&
	    info_hdr.iBitCount != 8  && info_hdr.iBitCount != 16 &&
	    info_hdr.iBitCount != 24 && info_hdr.iBitCount != 32) {
	    TIFFError(infilename,
		      "Cannot process BMP file with bit count %d",
		      info_hdr.iBitCount);
	    close(fd);
	    return NULL;
	}

	width = info_hdr.iWidth;
	length = (info_hdr.iHeight > 0) ? info_hdr.iHeight : -info_hdr.iHeight;

	bit = new_ibitmap(width, length, get_opt_ibitmap_depth(info_hdr.iBitCount));
	if (!bit) {
		TIFFError(infilename,
		"Can't allocate space for ibitmap");
		goto bad;
	}

	switch (info_hdr.iBitCount)
	{
		case 1:
		case 4:
		case 8:
			nbands = 1;
			depth = info_hdr.iBitCount;
			photometric = PHOTOMETRIC_PALETTE;
			/* Allocate memory for colour table and read it. */
			if (info_hdr.iClrUsed)
			    clr_tbl_size =
				    ((uint32)(1<<depth)<info_hdr.iClrUsed)
				    ? (uint32) (1 << depth)
				    : info_hdr.iClrUsed;
			else
			    clr_tbl_size = 1 << depth;
			clr_tbl = (unsigned char *)
				_TIFFmalloc(n_clr_elems * clr_tbl_size);
			if (!clr_tbl) {
				TIFFError(infilename,
				"Can't allocate space for color table");
				goto bad;
			}

			lseek(fd, BFH_SIZE + info_hdr.iSize, SEEK_SET);
			read(fd, clr_tbl, n_clr_elems * clr_tbl_size);

			break;
		case 16:
		case 24:
			nbands = 3;
			depth = info_hdr.iBitCount / nbands;
			photometric = PHOTOMETRIC_RGB;
			break;
		case 32:
			nbands = 3;
			depth = 8;
			photometric = PHOTOMETRIC_RGB;
			break;
		default:
			break;
	}

/* -------------------------------------------------------------------- */
/*  Read uncompressed image data.                                       */
/* -------------------------------------------------------------------- */

	if (info_hdr.iCompression == BMPC_RGB) {
		uint32 offset, size;
		char *scanbuf;

		/* XXX: Avoid integer overflow. We can calculate size
		 * in one step using
		 *
		 *  size = ((width * info_hdr.iBitCount + 31) & ~31) / 8
		 *
		 * formulae, but we should check for overflow
		 * conditions during calculation.
		 */
		size = width * info_hdr.iBitCount + 31;
		if (!width || !info_hdr.iBitCount
		    || (size - 31) / info_hdr.iBitCount != width ) {
			TIFFError(infilename,
				  "Wrong image parameters; can't "
				  "allocate space for scanline buffer");
			goto bad;
		}
		size = (size & ~31) / 8;

		scanbuf = (char *) _TIFFmalloc(size);
		if (!scanbuf) {
			TIFFError(infilename,
			"Can't allocate space for scanline buffer");
			goto bad;
		}

		for (row = 0; row < length; row++) {
			if (info_hdr.iHeight > 0)
				offset = file_hdr.iOffBits+(length-row-1)*size;
			else
				offset = file_hdr.iOffBits + row * size;
			if (lseek(fd, offset, SEEK_SET) == (off_t)-1) {
				TIFFError(infilename,
					  "scanline %lu: Seek error",
					  (unsigned long) row);
				break;
			}

			if (read(fd, scanbuf, size) < 0) {
				TIFFError(infilename,
					  "scanline %lu: Read error",
					  (unsigned long) row);
				break;
			}

			for (i = 0; i < width; i++) {
				switch (info_hdr.iBitCount) {
					case 4:
						r = (unsigned char)scanbuf[i>>1];
						r = g = b = ((i&1) == 1) ? ((r & 0x0f) + ((r << 4) & 0xf0)) :
						                           ((r & 0xf0) + ((r >> 4) & 0x0f)) ;
						break;
					case 8:
						r = g = b = (unsigned char)scanbuf[i];
						break;
					case 24:
					case 32:
						/* Skip the alpha channel if it exists. */
						a = (info_hdr.iBitCount == 32) ? 1 : 0;
						j = info_hdr.iBitCount/8;
						b = (unsigned char)scanbuf[i*j+a+0];
						g = (unsigned char)scanbuf[i*j+a+1];
						r = (unsigned char)scanbuf[i*j+a+2];
						break;
					default:
						/* 16-bit should be handled differently here. */
						r = g = b = 0;
						break;
				}
				setRGB(bit, i, row, r, g, b, 0);
			}
		}

		_TIFFfree(scanbuf);

/* -------------------------------------------------------------------- */
/*  Read compressed image data.                                         */
/* -------------------------------------------------------------------- */

	} else if ( info_hdr.iCompression == BMPC_RLE8
		    || info_hdr.iCompression == BMPC_RLE4 ) {
		uint32		i, j, k, runlength;
		uint32		compr_size, uncompr_size;
		unsigned char   *comprbuf;
		unsigned char   *uncomprbuf;

		compr_size = file_hdr.iSize - file_hdr.iOffBits;
		uncompr_size = width * length;
		comprbuf = (unsigned char *) _TIFFmalloc( compr_size );
		if (!comprbuf) {
			TIFFError(infilename,
			"Can't allocate space for compressed scanline buffer");
			goto bad;
		}
		uncomprbuf = (unsigned char *)_TIFFmalloc(uncompr_size);
		if (!uncomprbuf) {
			TIFFError(infilename,
			"Can't allocate space for uncompressed scanline buffer");
			goto bad;
		}

		lseek(fd, file_hdr.iOffBits, SEEK_SET);
		read(fd, comprbuf, compr_size);
		i = 0;
		j = 0;
		if (info_hdr.iBitCount == 8) {		/* RLE8 */
			while(j < uncompr_size && i < compr_size) {
				if ( comprbuf[i] ) {
					runlength = comprbuf[i++];
					while( runlength > 0
					   && j < uncompr_size
					   && i < compr_size ) {
						uncomprbuf[j++] = comprbuf[i];
						runlength--;
					}
					i++;
				} else {
					i++;
					if (comprbuf[i] == 0) /* Next scanline */
						i++;
					else if (comprbuf[i] == 1) /* End of image */
						break;
					else if (comprbuf[i] == 2) { /* Move to... */
						i++;
						if (i < compr_size - 1) {
							j+=comprbuf[i]+comprbuf[i+1]*width;
							i += 2;
						}
						else
							break;
			    } else {            /* Absolute mode */
						runlength = comprbuf[i++];
						for (k = 0; k < runlength && j < uncompr_size && i < compr_size; k++)
							uncomprbuf[j++] = comprbuf[i++];
						if ( k & 0x01 )
							i++;
					}
				}
			}
		}
		else {				    /* RLE4 */
			while( j < uncompr_size && i < compr_size ) {
				if ( comprbuf[i] ) {
					runlength = comprbuf[i++];
					while( runlength > 0 && j < uncompr_size && i < compr_size ) {
						if ( runlength & 0x01 )
							uncomprbuf[j++] = (comprbuf[i] & 0xF0) >> 4;
						else
							uncomprbuf[j++] = comprbuf[i] & 0x0F;
						runlength--;
					}
					i++;
				} else {
					i++;
					if (comprbuf[i] == 0) /* Next scanline */
						i++;
					else if (comprbuf[i] == 1) /* End of image */
						break;
			    else if (comprbuf[i] == 2) { /* Move to... */
						i++;
						if (i < compr_size - 1) {
							j+=comprbuf[i]+comprbuf[i+1]*width;
							i += 2;
						}
						else
							break;
					} else {            /* Absolute mode */
						runlength = comprbuf[i++];
						for (k = 0; k < runlength && j < uncompr_size && i < compr_size; k++) {
							if (k & 0x01)
								uncomprbuf[j++] = comprbuf[i++] & 0x0F;
							else
								uncomprbuf[j++] = (comprbuf[i] & 0xF0) >> 4;
						}
						if (k & 0x01)
							i++;
					}
				}
			}
		}

		_TIFFfree(comprbuf);

		for (row = 0; row < length; row++) {
			for (i = 0; i < width; i++) {
				clr = uncomprbuf[row*width+i];
				b = clr_tbl[clr*n_clr_elems];
				g = clr_tbl[clr*n_clr_elems+1];
				r = clr_tbl[clr*n_clr_elems+2];
				setRGB(bit, i, row, r, g, b, 0);
			}
		}

		_TIFFfree(uncomprbuf);
	}

	if (clr_tbl)
	  _TIFFfree(clr_tbl);
bad:
        close(fd);

        return bit;
}
