/*
 * gif2ibitmap() converts a GIF file to an ibitmap structure.
 * This code was hacked out from gif2tiff.c in tiff-3.9.2,
 * and the license in that file is reproduced below.
 */

/* $Id: gif2tiff.c,v 1.8 2004/09/02 14:36:33 dron Exp $ */

/*
 * Copyright (c) 1990-1997 Sam Leffler
 * Copyright (c) 1991-1997 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */

/*
 *	convert a GIF file into a TIFF file.
 *	based on Paul Haeberli's fromgif program which in turn is
 *	based on a GIF file reader by Marcel J.E. Mol March 23 1989 
 *
 *	if input is 320 by 200 pixel aspect is probably 1.2
 *	if input is 640 350 pixel aspect is probably 1.37
 *
 */
#include "tif_config.h"
#include "img2ibitmap.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#define	GIFGAMMA	(1.5)		/* smaller makes output img brighter */
#define	IMAX		0xffff		/* max intensity value */
#define EXTRAFUDGE	128		/* some people write BAD .gif files */

static unsigned short gamtab[256];

static void
makegamtab(float gam)
{
	int i;

	for(i=0; i<256; i++) 
		gamtab[i] = (unsigned short) (IMAX*pow(i/255.0,gam)+0.5);
}

#define COLSIZE 256

static unsigned char *stackp;
static unsigned int prefix[4096];
static unsigned char suffix[4096];
static unsigned char stack[4096];
static int datasize,codesize,codemask;     /* Decoder working variables */
static int clear,eoi;                      /* Special code values */
static int avail, oldcode;

static FILE *infile;
static int global;                        /* Is there a global color map? */
static int globalbits;                     /* Number of bits of global colors */
static unsigned char globalmap[COLSIZE][3];/* RGB values for global color map */
static unsigned char *raster;              /* Decoded image data */
static unsigned long width, height;
static unsigned short red[COLSIZE];
static unsigned short green[COLSIZE];
static unsigned short blue[COLSIZE];
static char *filename;
static ibitmap *bit = NULL;

static int	convert(void);
static int	checksignature(void);
static void	readscreen(void);
static int	readgifimage(char*);
static void	readextension(void);
static int	readraster(void);
static int	process(int, unsigned char**);
static void	initcolors(unsigned char [COLSIZE][3], int);
static void	rasterize(int, char*);


ibitmap* gif2ibitmap(const char* _filename)
{
	int status;

	makegamtab(GIFGAMMA);
	filename = (char*)_filename;
	if ((infile = fopen(filename, "rb")) == NULL) {
		perror(filename);
		return NULL;
	}
	status = convert();
	fclose(infile);
	return bit;
}

static int convert(void)
{
	int ch;
	char* mode = "w";

	if (!checksignature())
		return (-1);
	readscreen();
	while ((ch = getc(infile)) != ';' && ch != EOF) {
		switch (ch) {
			case '\0':  break;  /* this kludge for non-standard files */
			case ',':   if (!readgifimage(mode))
			            	return (-2);
			            mode = "a";		/* subsequent images append */
			            break;
			case '!':   readextension();
			            break;
			default:    fprintf(stderr, "illegal GIF block type\n");
			            	return (-1);
		}
	}
	return (0);
}

static int checksignature(void)
{
	char buf[6];

	fread(buf,1,6,infile);
	if (strncmp(buf,"GIF",3)) {
		fprintf(stderr, "file is not a GIF file\n");
		return 0;
	}
	/* Added "89a" but we don't support the animation features. */
	if (strncmp(&buf[3],"87a",3) && strncmp(&buf[3],"89a",3)) {
		fprintf(stderr, "unknown GIF version number\n");
		return 0;
	}
	return 1;
}

/*
 * 	readscreen - 
 *		Get information which is global to all the images stored 
 *	in the file
 */
static void readscreen(void)
{
	unsigned char buf[7];

	fread(buf,1,7,infile);
	global = buf[4] & 0x80;
	if (global) {
		globalbits = (buf[4] & 0x07) + 1;
		fread(globalmap,3,1<<globalbits,infile);
	}
}

static int readgifimage(char* mode)
{
	unsigned char buf[9];
	int local, interleaved;
	unsigned char localmap[256][3];
	int localbits;
	int status;

	if (fread(buf, 1, 9, infile) == 0) {
		perror(filename);
		return (0);
	}
	width = buf[4] + (buf[5] << 8);
	height = buf[6] + (buf[7] << 8);
	local = buf[8] & 0x80;
	interleaved = buf[8] & 0x40;

	if (local == 0 && global == 0) {
		fprintf(stderr, "no colormap present for image\n");
		return (0);
	}
 	if ((raster = (unsigned char*) malloc(width*height+EXTRAFUDGE)) == NULL) {
		fprintf(stderr, "not enough memory for image\n");
		return (0);
	}
	if (local) {
		localbits = (buf[8] & 0x7) + 1;

		fprintf(stderr, "   local colors: %d\n", 1<<localbits);

		fread(localmap, 3, 1<<localbits, infile);
		initcolors(localmap, 1<<localbits);
	} else if (global) {
		initcolors(globalmap, 1<<globalbits);
	}
	if ((status = readraster()))
		rasterize(interleaved, mode);
	free(raster);
	return status;
}

/*
 * 	readextension -
 *		Read a GIF extension block (and do nothing with it).
 *
 */
static void readextension(void)
{
	int count;
	char buf[255];

	(void) getc(infile);
	while ((count = getc(infile)))
		fread(buf, 1, count, infile);
}

/*
 * 	readraster -
 *		Decode a raster image
 *
 */
static int readraster(void)
{
	unsigned char *fill = raster;
	unsigned char buf[255];
	register int bits=0;
	register unsigned long datum=0;
	register unsigned char *ch;
	register int count, code;
	int status = 1;

	datasize = getc(infile);
	clear = 1 << datasize;
	eoi = clear + 1;
	avail = clear + 2;
	oldcode = -1;
	codesize = datasize + 1;
	codemask = (1 << codesize) - 1;
	for (code = 0; code < clear; code++) {
		prefix[code] = 0;
		suffix[code] = code;
	}
	stackp = stack;
	for (count = getc(infile); count > 0; count = getc(infile)) {
		fread(buf,1,count,infile);
		for (ch=buf; count-- > 0; ch++) {
			datum += (unsigned long) *ch << bits;
			bits += 8;
			while (bits >= codesize) {
				code = datum & codemask;
				datum >>= codesize;
				bits -= codesize;
				if (code == eoi) {               /* This kludge put in */
					goto exitloop;               /* because some GIF files*/
				}                                /* aren't standard */
				if (!process(code, &fill)) {
					status = 0;
					goto exitloop;
				}
			}
		}
		if (fill >= raster + width*height) {
			fprintf(stderr, "raster full before eoi code\n");
			break;
		}
	}
exitloop:
	if (fill != raster + width*height)  {
		fprintf(stderr, "warning: wrong rastersize: %ld bytes\n",
		        (long) (fill-raster));
		fprintf(stderr, "         instead of %ld bytes\n",
		        (long) width*height);
	}
	return status;
}

/*
 * 	process - 
 *		Process a compression code.  "clear" resets the code table.  
 *	Otherwise make a new code table entry, and output the bytes 
 *	associated with the code.
 */
static int process(register int code, unsigned char** fill)
{
	int incode;
	static unsigned char firstchar;

	if (code == clear) {
		codesize = datasize + 1;
		codemask = (1 << codesize) - 1;
		avail = clear + 2;
		oldcode = -1;
		return 1;
	}

	if (oldcode == -1) {
		*(*fill)++ = suffix[code];
		firstchar = oldcode = code;
		return 1;
	}
	if (code > avail) {
		fprintf(stderr, "code %d too large for %d\n", code, avail);
		return 0; 
	}

	incode = code;
	if (code == avail) {      /* the first code is always < avail */
		*stackp++ = firstchar;
		code = oldcode;
	}
	while (code > clear) {
		*stackp++ = suffix[code];
		code = prefix[code];
	}

	*stackp++ = firstchar = suffix[code];
	prefix[avail] = oldcode;
	suffix[avail] = firstchar;
	avail++;

	if (((avail & codemask) == 0) && (avail < 4096)) {
		codesize++;
		codemask += avail;
	}
	oldcode = incode;
	do {
		*(*fill)++ = *--stackp;
	} while (stackp > stack);
	return 1;
}

/*
 * 	initcolors -
 *		Convert a color map (local or global) to arrays with R, G and B
 * 	values. 
 *
 */
static void initcolors(unsigned char colormap[COLSIZE][3], int ncolors)
{
	register int i;

	for (i = 0; i < ncolors; i++) {
		red[i]   = gamtab[colormap[i][0]];
		green[i] = gamtab[colormap[i][1]];
		blue[i]  = gamtab[colormap[i][2]];
	}
}

void copy_pixel(unsigned long row, unsigned int col, unsigned char index) {
	setRGB(bit, col, row, red[index]>>8, green[index]>>8, blue[index]>>8, 0);
}

static void rasterize(int interleaved, char* mode)
{
	register unsigned long row;
	unsigned char *ras;
	unsigned int i;

	/* Assume that there are colours (hard to know...) */
	if ((bit = new_ibitmap(width, height, get_opt_ibitmap_depth(24))) == NULL) {
		fprintf(stderr, "not enough memory for image\n");
		return;
	}

#define DRAWSEGMENT(offset, step) {			\
		for (row = offset; row < height; row += step) {	\
			for (i = 0; i < width; i++) copy_pixel(row, i, ras[i]); \
			ras += width;                            	\
		}						\
	}
	ras = raster;
	if (interleaved) {
		DRAWSEGMENT(0, 8);
		DRAWSEGMENT(4, 8);
		DRAWSEGMENT(2, 4);
		DRAWSEGMENT(1, 2);
	} else 
		DRAWSEGMENT(0, 1);
#undef DRAWSEGMENT

}

