#include "img2ibitmap.h"

static int _max_depth = 8;	/* This can be set higher by the application. */
static int _min_depth = 4;	/* This cannot be changed. */

ibitmap* new_ibitmap(unsigned short width, unsigned short height,
                     unsigned short depth) {
	unsigned short scanline = (unsigned short)((((int)width*depth + 31) & ~31)/8);  //Round to 4-bytes
	ibitmap *bm = (ibitmap*)malloc(sizeof(ibitmap) + height*scanline);
	if (bm) {
		bm->width = width;
		bm->height = height;
		bm->depth = depth;
		bm->scanline = scanline;
	}
	return bm;
}

int get_max_ibitmap_depth() {
	return _max_depth;
}

void set_max_ibitmap_depth(int d) {
	_max_depth = (d > _min_depth) ? d : _min_depth;
}

int get_opt_ibitmap_depth(int input_depth) {
	/* Choose the optimal ibitmap depth for the device. */
	return (input_depth > _max_depth) ? _max_depth :
	       (input_depth < _min_depth) ? _min_depth : input_depth;
}

unsigned char rgb2gs(unsigned char r, unsigned char g, unsigned char b) {
	return (unsigned char)((117*(int)b + 601*(int)g + 306*(int)r + 512) >> 10);
}

#define VALID_PIXEL(bm, x, y) \
	(bm && ((x) >= 0) && ((x) < bm->width) && ((y) >= 0) && ((y) < bm->height))

int getRGB(ibitmap *bm, int x, int y, unsigned char *r,
           unsigned char *g, unsigned char *b, int skip_check) {
	int ret = 0, a, s;

	if (skip_check || VALID_PIXEL(bm, x, y)) {
		switch (bm->depth) {
			case 4:
				*r = *g = *b = getBrightness(bm, x, y, 1);
				break;
			case 8:
				*r = *g = *b = bm->data[y*bm->scanline + x];
				break;
			case 24:
			case 32:
				/* Assume (A)BGR order. */
				a = (bm->depth == 32) ? 1 : 0; /* Account for the alpha channel. */
				s = bm->depth/8;
				*b = bm->data[y*bm->scanline + x*s + a + 0];
				*g = bm->data[y*bm->scanline + x*s + a + 1];
				*r = bm->data[y*bm->scanline + x*s + a + 2];
				break;
			default:
				*r = *g = *b = 0;
				ret = -1;
				break;
		}
	} else
		ret = -1;

	return ret;
}

int setRGB(ibitmap *bm, int x, int y, unsigned char r,
           unsigned char g, unsigned char b, int skip_check) {
	int ret = 0, a, s;
	unsigned char oldpair, newval;

	if (skip_check || VALID_PIXEL(bm, x, y)) {
		switch (bm->depth) {
			case 4:
				oldpair = bm->data[y*bm->scanline + (x>>1)];
				newval = rgb2gs(r, g, b) >> 4;
				bm->data[y*bm->scanline + (x>>1)] = ((x & 1) == 0) ?
				                                    ((oldpair & 0x0f) + (newval<<4)) :
				                                    ((oldpair & 0xf0) + newval) ;
				break;
			case 8:
				bm->data[y*bm->scanline + x] = rgb2gs(r, g, b);
				break;
			case 24:
			case 32:
				/* Assume (A)BGR order. */
				a = (bm->depth == 32) ? 1 : 0; /* Skip the alpha channel. */
				s = bm->depth/8;
				bm->data[y*bm->scanline + x*s + a + 0] = b;
				bm->data[y*bm->scanline + x*s + a + 1] = g;
				bm->data[y*bm->scanline + x*s + a + 2] = r;
				break;
			default:
				ret = -1;
				break;
		}
	} else
		ret = -1;

	return ret;
}

unsigned char getBrightness(ibitmap *bm, int x, int y, int skip_check) {
	unsigned char r, g, b, brightness = 0;

	if (skip_check || VALID_PIXEL(bm, x, y)) {
		switch (bm->depth) {
			case 4:
				r = bm->data[y*bm->scanline + (x>>1)];
				brightness = ((x&1) == 1) ? ((r & 0x0f) + ((r << 4) & 0xf0)) :
				                            ((r & 0xf0) + ((r >> 4) & 0x0f)) ;
				break;
			case 8:
				brightness = bm->data[y*bm->scanline + x];
				break;
			case 16:
			case 24:
			case 32:
				if (getRGB(bm, x, y, &r, &g, &b, 1) != -1)
					brightness = rgb2gs(r, g, b);
				break;
			default:
				break;
		}
	}

	return brightness;
}
