#include <stdio.h>	/* For fopen() fseek() fread() fwrite fclose() */
#include <unistd.h>	/* For unlink() */
#include <errno.h>
#include <inkview.h>	/* For Message() */
#include "C7zHandler.h"
#include "ImagesMng.h"
#include "Log.h"

/*
 * These input and output stream classes are required by 7zlib.
 */

class PBInStream : public C7ZipInStream {
private:
	FILE *_f;
	std::string _path;
	wstring _ext;
	int _size;
public:
	PBInStream(std::string path) : _path(path), _ext(L"7z") {
		if ((_f = fopen(path.c_str(), "rb"))) {
			fseek(_f, 0 , SEEK_END);
			_size = ftell(_f);
			fseek(_f, 0 , SEEK_SET);
			size_t pos = _path.find_last_of(".");
			if (pos != std::string::npos) {
				std::string ext = _path.substr(pos + 1);
				_ext = wstring(ext.begin(), ext.end());
			}
		} else 
			_size = -errno;
	}

	virtual ~PBInStream() {
		fclose(_f);
	}

	virtual wstring GetExt() const {
		return _ext;
	}

	virtual int Read(void *data, unsigned int size, unsigned int *processedSize) {
		int ret = 1, count;

		if (!_f)
			return 1;

		if ((count = fread(data, 1, size, _f)) >= 0) {
			if (processedSize)
				*processedSize = count;
			ret = 0;
		}
		Log::msg("Read: size = %d, count = %d", size, count);

		return ret;
	}

	virtual int Seek(__int64 offset, unsigned int seekOrigin, unsigned __int64 *newPosition) {
		int ret = 1;

		if (!_f)
			return 1;

		Log::msg("RSeek: offset = %lld, origin = %u, pos = %lld", offset, seekOrigin, newPosition ? *newPosition : -1);
		if (!(fseek(_f, (long)offset, seekOrigin))) {
			if (newPosition)
				*newPosition = ftell(_f);
			ret = 0;
		}

		return ret;
	}

	virtual int GetSize(unsigned __int64 *size) {
		if (size)
			*size = _size;
		return 0;
	}
};

class PBOutStream : public C7ZipOutStream {
private:
	FILE *_f;
	std::string _path;
	wstring _ext;
	int _size;
public:
	PBOutStream(std::string path) : _path(path), _ext(L"7z") {
		if ((_f = fopen(path.c_str(), "wb"))) {
			size_t pos = _path.find_last_of(".");
			if (pos != std::string::npos) {
				std::string ext = _path.substr(pos + 1);
				_ext = wstring(ext.begin(), ext.end());
			}
			_size = 0;
		} else
			_size = -errno;
	}

	virtual ~PBOutStream() {
		fclose(_f);
	}

	int GetFileSize() const {
		return _size;
	}

	virtual int Write(const void *data, unsigned int size, unsigned int *processedSize) {
		int ret = 1, count;

		if ((count = fwrite(data, 1, size, _f)) >= 0) {
			if (processedSize)
				*processedSize = count;
			_size += count;
			ret = 0;
		}
		Log::msg("Write: size = %d, count = %d", size, count);

		return ret;
	}

	virtual int Seek(__int64 offset, unsigned int seekOrigin, unsigned __int64 *newPosition) {
		int ret = 1;

		Log::msg("WSeek: offset = %lld, origin = %u, pos = %lld", offset, seekOrigin, newPosition ? *newPosition : -1);
		if (!(fseek(_f, (long)offset, seekOrigin))) {
			if (newPosition)
				*newPosition = ftell(_f);
			ret = 0;
		}

		return ret;
	}

	virtual int SetSize(unsigned __int64 size) {
		return 0;
	}
};


C7zHandler::C7zHandler(std::string& outdir) : ArchiveHandler(outdir) {
	_arch = NULL;
	_in = NULL;
  if (!_lib.Initialize())
		Log::msg("C7z Init failed");
}

C7zHandler::~C7zHandler() {
	if (_arch) {
		_arch->Close();
		delete _arch;
	}
	if (_in)
		delete _in;
	_lib.Deinitialize();
}

#if 0
bool C7zHandler::IsSupportedZipImage(const char *path) {
	/* If the file is unencrypted, get the type from the magic number.
	 * Otherwise, guess based on the name. */

	struct zip_stat sb;
	struct zip_file *zf;
	char file_magic[] = { 0, 0, 0, 0, 0, 0, 0, 0 };

	if (zip_stat(_zip, path, 0, &sb) == 0) {
		if (sb.encryption_method == ZIP_EM_NONE) {
			if ((zf = zip_fopen_index(_zip, sb.index, 0))) {
				zip_fread(zf, file_magic, (sb.size < 8) ? sb.size : 8);
				zip_fclose(zf);
				return (ImagesMng::GetMagicType(file_magic) != ImagesMng::UNSUPPORTED_IMAGE);
			}
		}
	}

	return ImagesMng::IsSupportedImage(path);
}
#endif

int C7zHandler::Open(std::string& path, std::vector<std::string>& files) {
	int failed = 0, err = 0;
	unsigned __int64 size;
	unsigned int numitems;
	C7ZipArchiveItem *item;
	int buflen = 1024;
	char errmsg[buflen];
	char name[buflen];
	bool usemagic;

	usemagic = !(ImagesMng::HasSuffix(path.c_str(), ".tar") ||
	             ImagesMng::HasSuffix(path.c_str(), ".cbt"));
	_in = new PBInStream(path);
	_in->GetSize(&size);
	if ((__int64)size < 0) {
		err = -(int)((__int64)size);
	} else if (!(_lib.OpenArchive(_in, &_arch, usemagic))) {
		err = _lib.GetLastError();
	}

	if (!err) {
		files.clear();

		_arch->GetItemCount(&numitems);
		for (int i = 0; i < (int)numitems; i++) {
			if (_arch->GetItemInfo(i, &item) && !item->IsDir()) {
				if (wcstombs(name, item->GetFullPath().c_str(), buflen) >= 0) {
					/* Don't know how to read just the first few bytes of a file,
					 * so we don't check the magic number here. */
					if (ImagesMng::IsSupportedImage(name))
						files.push_back(name);
				}
			}
		}
	} else {
		if (snprintf(errmsg, buflen,
		             "Open7z: couldn't open archive %s: error = %d\n",
		             path.c_str(), err) >= buflen)
			errmsg[buflen-1] = '\0';	/* Force NULL termination */
		Message(ICON_ERROR, GetLangText("@Program_name"), errmsg, 3000);
		failed = 1;
	}
	return failed;
}

int C7zHandler::UnpackFile(std::string& name, std::string &outname) {
	int need_passwd = 0;
	int size;
	unsigned int numitems;
	C7ZipArchiveItem *item;
	int buflen = 1024;
	char buf[buflen];

	if (_arch) {
		if (_passwd.size() && !_arch->IsPasswordSet()) {
			wchar_t *passwd = NULL;
			if (mbstowcs(passwd, _passwd.c_str(), 0) >= 0) {
				_arch->SetArchivePassword(wstring(passwd));
				free(passwd);
			}
		}
		outname = _dir;
		size_t slash_pos = name.find_last_of("/");
		if (slash_pos != std::string::npos)
			outname += name.substr(slash_pos + 1);
		else
			outname += name;
		unlink(outname.c_str());  /* Get rid of any old copies. */

		_arch->GetItemCount(&numitems);
		for (int i = 0; i < (int)numitems; i++) {
			if (_arch->GetItemInfo(i, &item) && !item->IsDir()) {
				Log::msg("Trying i = %d", i);
				if (wcstombs(buf, item->GetFullPath().c_str(), buflen) >= 0) {
					if (name == buf) {
						if (item->IsEncrypted() && !_passwd.size()) {
							outname = "";   // Need a password, but don't have one
							need_passwd = 1;
						} else {
							PBOutStream out = PBOutStream(outname.c_str());
							size = out.GetFileSize();
							if (size < 0) {
								if (snprintf(buf, buflen,
								             "Unpack7zFile: couldn't open output file %s: error = %d\n",
								             outname.c_str(), -size) >= buflen)
									buf[buflen-1] = '\0';	/* Force NULL termination */
								Message(ICON_ERROR, GetLangText("@Program_name"), buf, 3000);
							//} else if (!_arch->Extract(i, &out)) {
							} else if (!_arch->Extract(item, &out)) {
								outname = "";
								if (_lib.GetLastError() == lib7zip::LIB7ZIP_NEED_PASSWORD) {
									Message(ICON_ERROR, GetLangText("@Program_name"), GetLangText("@PasswordNotMatch"), 3000);
									need_passwd = 1;
								}
							}
						}
						break;
					}
				}
			}
		}
	} else
		Message(ICON_ERROR, GetLangText("@Program_name"), "Unpack7zFile", 3000);
	return need_passwd;
}
