/*
 * Viewer.cpp
 *
 *  Created on: 26.01.2010
 *      Author: Admin
 */

#include <img2ibitmap.h>
#include "Viewer.h"

static bool _l2r;	//Flag for left-to-right panel mode, used in panel sort.

Viewer::Viewer()
{
	SetFitMode(FIT_NOT);
	SetScaleline((double*) DEFSCALELINE);
	SetClipRect((irect*) &DEFCLIPRECT);
	Zoom100();
	CalcFitScale();
	SetMoveStep(MOVESPET_SCREEN);
	MoveAlign(ALIGN_CENTER | VALIGN_MIDDLE);
	_do_auto_panels = false;
	_do_scale_panels = false;
	_readdir = DIRECTION_RIGHT;
	_bkgd_threshold = 220;
	_do_crop = false;
	_marginL = _marginR = _marginT = _marginB = 0;
	_margin_size = 4;
	_margin_thresh = 2;
}

Viewer::~Viewer()
{
	_panels.clear();
}

void Viewer::Draw()
{
	if (_image != NULL)
	{
		Log::msg("Width: %d, height: %d, depth: %d, scale: %.2f", _image->width, _image->height, _image->depth,
				GetScale());
		icanvas* c = GetCanvas();
		Log::msg("x: %d, y: %d, x2: %d, y2: %d", c->clipx1, c->clipy1, c->clipx2, c->clipy2);
		StretchBitmap(_imagerect.x, _imagerect.y, _imagerect.w, _imagerect.h, _image, 0);
		if (_do_panels)
			DrawPanelRect();
	}
	else
		Log::msg("Current is NULL!");
}

void Viewer::SetClipRect(int x, int y, int w, int h)
{
	_cliprect.x = x;
	_cliprect.y = y;
	_cliprect.w = w;
	_cliprect.h = h;
}

bool Viewer::CheckForNewImage(ibitmap* bit, bool update_testpoints) {
	/* This returns true if the image not the same as the last one.
	 * It checks the ibitmap pointer values, and four pixels on the images. */
	bool is_new = false;
	int _w = bit->width, _h = bit->height;
	int x1 = _w/4, x2 = 3*_w/4;
	int y1 = _h/4, y2 = 3*_h/4;
	unsigned char i11 = getBrightness(bit, x1, y1, 0),
	              i12 = getBrightness(bit, x1, y2, 0),
	              i21 = getBrightness(bit, x2, y1, 0),
	              i22 = getBrightness(bit, x2, y2, 0);
	if ((bit != _image) || (!_image) ||
	    (_w != _image->width) || (_h != _image->height) ||
	    !((i11 == testpoints[0]) && (i12 == testpoints[1]) &&
	      (i21 == testpoints[2]) && (i22 == testpoints[3]))) {
		is_new = true;
		if (update_testpoints) {
			testpoints[0] = i11;
			testpoints[1] = i12;
			testpoints[2] = i21;
			testpoints[3] = i22;
		}
	}

	return is_new;
}

void Viewer::SetImage(ibitmap* bit)
{
	if (bit && CheckForNewImage(bit, true)) {
		/* If the image changed, clear the old panel information. */
		_panels.clear();
		_do_panels = _do_auto_panels;
	}
	if (_fitmode != FIT_CUSTOM) {
		_scale = 1.0;
		_fitscale = 1.0;
	}
	_imagerect = _cliprect;
	_image = bit;
	CalcMargins();
	CalcFitScale();
	ZoomFit();
	if (CroppedW() <= _cliprect.w)
		MoveAlign(ALIGN_CENTER);
	else
		MoveAlign((_readdir == DIRECTION_RIGHT) ? ALIGN_LEFT : ALIGN_RIGHT);
	if (CroppedH() <= _cliprect.h)
		MoveAlign(VALIGN_MIDDLE);
	else
		MoveAlign(VALIGN_TOP);
	if (_do_panels)
		EnablePanelMode();
}

void Viewer::CalcMargins()
{
	Log::msg("size = %d, thresh = %d", _margin_size, _margin_thresh);
	if (_image != NULL && _do_crop) {
		int x, y, n, found;
		found = 0;
		for (y = 0; y < _image->height && !found; y++) {
			n = 0;
			for (x = 0; x < _image->width && !found; x++) {
				n = (isBackground(x, y)) ? 0 : n + 1;
				if (n  > _margin_thresh) {
					_marginT = y - _margin_size;
					if (_marginT < 0) _marginT = 0;
					found = 1;
				}
			}
		}
		found = 0;
		for (y = 0; y < _image->height && !found; y++) {
			n = 0;
			for (x = 0; x < _image->width && !found; x++) {
				n = (isBackground(x, _image->height - y - 1)) ? 0 : n + 1;
				if (n > _margin_thresh) {
					_marginB = y - _margin_size;
					if (_marginB < 0) _marginB = 0;
					found = 1;
				}
			}
		}
		found = 0;
		for (x = 0; x < _image->width && !found; x++) {
			n = 0;
			for (y = 0; y < _image->height && !found; y++) {
				n = (isBackground(x, y)) ? 0 : n + 1;
				if (n > _margin_thresh) {
					_marginL = x - _margin_size;
					if (_marginL < 0) _marginL = 0;
					found = 1;
				}
			}
		}
		found = 0;
		for (x = 0; x < _image->width && !found; x++) {
			n = 0;
			for (y = 0; y < _image->height && !found; y++) {
				n = (isBackground(_image->width - x - 1, y)) ? 0 : n + 1;
				if (n > _margin_thresh) {
					_marginR = x - _margin_size;
					if (_marginR < 0) _marginR = 0;
					found = 1;
				}
			}
		}
	} else
		_marginL = _marginR = _marginT = _marginB = 0;
	Log::msg("l = %d, r = %d, t = %d, b = %d", _marginL, _marginR, _marginT, _marginL);
}

void Viewer::SetScale(double scale)
{
	double rescale = scale / _scale;
	int dx, dy;
	_scale = scale;
	if (_image != NULL)
	{
		_imagerect.w = (int) (_image->width * scale);
		_imagerect.h = (int) (_image->height * scale);
		dx = (int) ((_cliprect.x + _cliprect.w/2 - _imagerect.x)*(rescale - 1.0));
		dy = (int) ((_cliprect.y + _cliprect.h/2 - _imagerect.y)*(rescale - 1.0));
		Move(dx, dy);
		if (CroppedW() <= _cliprect.w && CroppedH() <= _cliprect.h)
		{
			MoveAlign(VALIGN_MIDDLE | ALIGN_CENTER);
		}
		else
		{
			Move(0, 0); // заглушка для корректировки выхода за границы изображения
			// TODO: нормальный расчет центральной точки при увеличении и уменьшении
			// int vx = (int)((_cliprect.w / 2 + _cliprect.x - _imagerect.x) * (scale / _scale - 1));
			// int vy = (int)((_cliprect.h / 2 + _cliprect.y - _imagerect.y) * (scale / _scale - 1));
			// Move(-vx, -vy);
			// Log::msg("vx %d, vy %d",-vx,-vy);
			if (CroppedW() <= _cliprect.w)
				MoveAlign(ALIGN_CENTER);
			if (CroppedH() <= _cliprect.h)
				MoveAlign(VALIGN_MIDDLE);
			if (_do_panels)
				AlignPanel();
		}
	}
	else
		_scale = scale;
}

bool Viewer::CanZoomIn()
{
	if (_scaleline != NULL)
	{
		if (_scale < _fitscale)
			return true;
		int i = 0;
		while (_scaleline[i] != 0)
			i++;
		if (_scale < _scaleline[i-1])
			return true;
	}
	return false;
}

bool Viewer::CanZoomOut()
{
	if (_scaleline != NULL && (_scale > _fitscale || _scale > _scaleline[0]))
		return true;
	return false;
}

void Viewer::ZoomIn() //TODO: Сделать правильное увеличение для масштаба заполнения большего, чем максимум в линейке
{
	if (_scaleline != NULL)
	{
		for (unsigned int i = 0; _scaleline[i] != 0; i++)
		{
			if (_scaleline[i] <= _scale)
				continue;
			if (_scale < _fitscale && _fitscale < _scaleline[i])
				SetScale(_fitscale);
			else
				SetScale(_scaleline[i]);
			break;
		}
	}
}

void Viewer::ZoomOut()
{
	if (_scaleline != NULL)
	{
		double temp = 0;
		for (unsigned int i = 0; _scaleline[i] != 0; i++)
		{
			if (_scaleline[i] < _scale)
			{
				temp = _scaleline[i];
				continue;
			}
			if (_scale > _fitscale && _fitscale > temp)
				SetScale(_fitscale);
			else
				SetScale(temp);
			break;
		}
	}
}

void Viewer::Zoom(int num)
{
	if (_scaleline != NULL)
		for (int i = 0; _scaleline[i] != 0; i++)
			if (i == num)
			{
				SetScale(_scaleline[num]);
				break;
			}
}

void Viewer::CalcFitScale()
{
	if (_fitmode == FIT_CUSTOM)
		return;

	if (_image != NULL) {
		int croppedWidth = _image->width - _marginL - _marginR;
		int croppedHeight = _image->height - _marginT - _marginB;
		if (_fitmode == FIT_WIDTH || (_fitmode == (FIT_WIDTH | FIT_OVERSIZE) && croppedWidth > _cliprect.w))
			_fitscale = (_cliprect.w / (double) croppedWidth);
		else if (_fitmode == FIT_HEIGHT ||
		         (_fitmode == (FIT_HEIGHT | FIT_OVERSIZE) && croppedHeight > _cliprect.h))
			_fitscale = (_cliprect.h / (double) croppedHeight);
		else if (_fitmode == FIT_BOTH ||
		         (_fitmode == (FIT_BOTH | FIT_OVERSIZE) &&
		          (croppedWidth > _cliprect.w || croppedHeight > _cliprect.h)))
			_fitscale = ((double)croppedWidth/_cliprect.w > (double)croppedHeight/_cliprect.h ?
			            _cliprect.w / (double) croppedWidth :
			            _cliprect.h / (double) croppedHeight);
		else
			_fitscale = (1.0);
	} else
		_fitscale = (1.0);
}

bool Viewer::CanMove(int dx, int dy)
{
	if (dx == 0 || (dx < 0 && CroppedLPos() <= _cliprect.x + dx) ||
			(dx > 0 && CroppedRPos() >= _cliprect.x + _cliprect.w + dx))
		if (dy == 0 || (dy < 0 && CroppedTPos() <= _cliprect.y + dy) ||
		    (dy > 0 && CroppedBPos() >= _cliprect.y + _cliprect.h + dy))
			return true;
	return false;
}

void Viewer::Move(int dx, int dy)
{
	_imagerect.x -= dx;
	_imagerect.y -= dy;
	if (CroppedLPos() > _cliprect.x &&
			CroppedW() > _cliprect.w)
		MoveAlign(ALIGN_LEFT);
	if (CroppedRPos() < _cliprect.x + _cliprect.w &&
			CroppedW() > _cliprect.w)
		MoveAlign(ALIGN_RIGHT);
	if (CroppedTPos() > _cliprect.y &&
			CroppedH() > _cliprect.h)
		MoveAlign(VALIGN_TOP);
	if (CroppedBPos() < _cliprect.y + _cliprect.h &&
			CroppedH() > _cliprect.h)
		MoveAlign(VALIGN_BOTTOM);
}

bool Viewer::CanMoveDirection(int direction)
{
	// XXXXX up/down may still show possible move even though there are no
	// panels to move to underneath.
	bool canmoveright = false, canmoveleft = false;
	if (direction & DIRECTION_UP && CanMove(0, -1))
		return true;
	if (direction & DIRECTION_DOWN && CanMove(0, 1))
		return true;
	if (direction & DIRECTION_LEFT)
		canmoveleft = _do_panels ? (GetNextPanel(direction) != -1)
		                         : CanMove(-1, 0);
	if (direction & DIRECTION_RIGHT)
		canmoveright = _do_panels ? (GetNextPanel(direction) != -1)
		                          : CanMove(1, 0);
	return (canmoveleft | canmoveright);
}

void Viewer::MoveDirection(int direction)
{
	if (CanMoveDirection(direction))
	{
		if (_do_panels)
			MovePanel(direction);
		else {
			int dx = 0, dy = 0;
			if (direction & DIRECTION_UP)
				dy -= (int) (_cliprect.h * _movestep);
			if (direction & DIRECTION_DOWN)
				dy += (int) (_cliprect.h * _movestep);
			if (direction & DIRECTION_LEFT)
				dx -= (int) (_cliprect.w * _movestep);
			if (direction & DIRECTION_RIGHT)
				dx += (int) (_cliprect.w * _movestep);
			Move(dx, dy);
		}
	}
}

void Viewer::MoveAlign(int align)
{
	if (align & ALIGN_LEFT)
		_imagerect.x = _cliprect.x - LeftMargin();
	else if (align & ALIGN_CENTER)
		_imagerect.x = _cliprect.x + (_cliprect.w - CroppedW()) / 2 - LeftMargin();
	else if (align & ALIGN_RIGHT)
		_imagerect.x = _cliprect.x + _cliprect.w - _imagerect.w + RightMargin();
	if (align & VALIGN_TOP)
		_imagerect.y = _cliprect.y - TopMargin();
	else if (align & VALIGN_MIDDLE)
		_imagerect.y = _cliprect.y + (_cliprect.h - CroppedH()) / 2 - TopMargin();
	else if (align & VALIGN_BOTTOM)
		_imagerect.y = _cliprect.y + _cliprect.h - _imagerect.h + BottomMargin();
}

bool Viewer::SetReadingDirection(int direction) {
	/* Sets the direction and returns true if the setting changed. */
	if (_readdir != direction) {
		_readdir = direction;
		if (_do_panels)
			EnablePanelMode();	/* Recompute the panel order */
		return true;
	} else
		return false;
}

// The rest of the code is for handling cartoon panels.

void Viewer::DisplayPanel() {
	/* Sets up the image to display the current panel, changing
	 * the scale if necessary. */
	if (_do_scale_panels) {
		int margin = 8;
		irect s = _panels[_current_panel];
		double xscale = (double)(_cliprect.w-margin) / s.w;
		double yscale = (double)(_cliprect.h-margin) / s.h;
		SetScale(xscale < yscale ? xscale : yscale);
	} else
		AlignPanel();
}

void Viewer::AlignPanel() {
	/* Aligns the current panel on the screen. */
	int dx = 0, dy = 0;

	if (!_panels.size())
		return;

	/* First, compute the distance to the center of the current panel
	 * from the current screen center position. */
	irect s = _panels[_current_panel];
	int xpanelc = (int)((s.x + s.w/2)*_scale);
	int ypanelc = (int)((s.y + s.h/2)*_scale);
	int xdisplayc = _cliprect.x + _cliprect.w/2 - _imagerect.x;
	int ydisplayc = _cliprect.y + _cliprect.h/2 - _imagerect.y;
	dx = xpanelc - xdisplayc;
	dy = ypanelc - ydisplayc;

	//Limit the movement to keep as much of the image as possible in the cliprect.
	int offsetx = (_cliprect.w > CroppedW()) ? (_cliprect.w-CroppedW())/2 : 0;
	int offsety = (_cliprect.h > CroppedH()) ? (_cliprect.h-CroppedH())/2 : 0;
	if (CroppedLPos()-dx > _cliprect.x+offsetx)
		dx = CroppedLPos() - _cliprect.x - offsetx;
	if (CroppedRPos()-dx < _cliprect.x+_cliprect.w-offsetx)
		dx = CroppedRPos() - _cliprect.x - _cliprect.w + offsetx;
	if (CroppedTPos()-dy > _cliprect.y+offsety)
		dy = CroppedTPos() - _cliprect.y - offsety;
	if (CroppedBPos()-dy < _cliprect.y+_cliprect.h-offsety)
		dy = CroppedBPos() + _cliprect.y - _cliprect.h + offsety;

	Move(dx, dy);
}

int Viewer::GetNextPanel(int direction) {
	/* Given the desired direction, the index to the next panel is returned. */
	int panel = -1, i, numpanels = _panels.size();

	if ((direction == DIRECTION_LEFT) || (direction == DIRECTION_RIGHT)) {
		/* Look for next panel that isn't already entirely onscreen */
		int inc = (direction == DIRECTION_LEFT) ? (_l2r ? -1 : 1)
		                                        : (_l2r ? 1 : -1);
		for (i = _current_panel + inc; (i >= 0 && i < numpanels); i += inc) {
			irect r = _panels[i];
			int left = (int)(r.x*_scale) + _imagerect.x;
			int top = (int)(r.y*_scale) + _imagerect.y;
			int right = (int)((r.x+r.w)*_scale) + _imagerect.x;
			int bottom = (int)((r.y+r.h)*_scale) + _imagerect.y;
			if ((left < _cliprect.x) || (right > (_cliprect.x + _cliprect.w)) ||
			    (top < _cliprect.y) || (bottom > (_cliprect.y + _cliprect.h))) {
				panel = i;
				break;
			}
		}
	} else if ((direction == DIRECTION_UP) || (direction == DIRECTION_DOWN)) {
		/* Find the panel above or below whose center is closest to the
		 * current panel. */
		int tmin = _image->width * _image->height;  //Something big
		irect r = _panels[_current_panel];
		int xc = r.x + r.w/2;
		int yc = r.y + r.h/2;
		for (i = 0; i < numpanels; i++) {
			if (i == _current_panel)
				continue;
			irect s = _panels[i];
			int xcs = s.x + s.w/2;
			int ycs = s.y + s.h/2;
			if (((direction == DIRECTION_UP) && (ycs < r.y)) ||
			    ((direction == DIRECTION_DOWN) && (ycs > r.y+r.h)))  {
				int dist = (int) sqrt((double)(xc-xcs)*(xc-xcs) +
				                      (double)(yc-ycs)*(yc-ycs));
				if (dist < tmin) {
					tmin = dist;
					panel = i;
				}
			}
		}
	}
	if ((panel < 0) || (panel >= numpanels))
		panel = -1;

	return panel;
}

void Viewer::MovePanel(int direction) {
	int panel = GetNextPanel(direction);
	if (panel != -1) {
		_current_panel = panel;
		DisplayPanel();
	}
}

bool Viewer::isBackground(int x, int y) {
	return getBrightness(_image, x, y, 0) > _bkgd_threshold;
}

irect Viewer::FloodFill(int x, int y, unsigned char newVal,
                           bool skipBkgdCheck) {
	/* Find all the neighbouring points in the mask with with the same
	 * value as the one at (x,y), and replace all of their mask values
	 * with 'newVal'.
	 * If 'skipBkgdCheck' is false, then the corresponding pixel in the
	 * image must also be a background colour to be changed.
	 * The bounding rectangle about the group of changed points is returned. */

	/* Got this algorithm from Lode Vandevenne's website:
	 * http://lodev.org/cgtutor/floodfill.html */

	int _w = _image->width, _h = _image->height;
	unsigned char oldVal = mask[y*_w + x];
	irect r = { x, y, 1, 1, 0 };
	std::vector<int> s;
	bool spanUp, spanDown;
	int i;

	s.clear();
	s.push_back(y*_w + x);

#define test(x, y) ((mask[(y)*_w+(x)] == oldVal) && \
                    (skipBkgdCheck || isBackground((x), (y))))

	while (!s.empty()) {
		i = s.back();
		s.pop_back();
		x = i % _w;
		y = i / _w;
		while ((x >= 0) && test(x, y)) x--;
		x++;
		spanUp = spanDown = false;
		while ((x < _w) && test(x, y)) {
			mask[y*_w + x] = newVal;
			if (x < r.x) {
				r.w += r.x - x;
				r.x = x;
			}
			if (x >= (r.x+r.w)) r.w = x - r.x + 1;
			if (y < r.y) {
				r.h += r.y - y;
				r.y = y;
			}
			if (y >= (r.y+r.h)) r.h = y - r.y + 1;
			if (!spanUp && (y > 0) && test(x, y-1)) {
				s.push_back((y-1)*_w + x);
				spanUp = true;
			} else if (spanUp && (y > 0) && !test(x, y-1)) {
				spanUp = false;
			}
			if (!spanDown && (y < _h-1) && test(x, y+1)) {
				s.push_back((y+1)*_w + x);
				spanDown = true;
			} else if (spanDown && (y < _h-1) && !test(x, y+1)) {
				spanDown = false;
			}
			x++;
		}
	}

#undef test

	return r;
}

void Viewer::StartBkgdSearchAt(int x, int y, int xstep, int ystep) {
	/* Starting at the given point, step along until a background pixel is
	 * found, and then mask out it and its neighbouring background pixels. */

	int _w = _image->width, _h = _image->height;

	while ((x>=0) && (x<_w) && (y>=0) && (y<_h) && (mask[y*_w+x] == 0)) {
		if (isBackground(x, y)) {
			irect r = FloodFill(x, y, 1, false);
			return;
		} else {
			x += xstep;
			y += ystep;
		}
	}
}

void Viewer::ComputeBackground() {
	/* Starting from the center of each edge of the image, mask out
	 * where neighbouring background pixels are found. */

	int _w = _image->width, _h = _image->height;
	int masksize = _w * _h;
	memset(mask, 0, masksize);

	StartBkgdSearchAt(_w/2, 0, 0, 1);
	StartBkgdSearchAt(0, _h/2, 1, 0);
	StartBkgdSearchAt(_w-1, _h/2, -1, 0);
	StartBkgdSearchAt(_w/2, _h-1, 0, -1);
}

struct Panel_sort {
	/* This is for sorting the reading order of the panels. */

	bool operator()(const irect& a, const irect& b) const {
		//Determine if the two panels are aligned horizontally
		int y0 = a.y > b.y ? a.y : b.y;
		int y1 = (a.y+a.h) < (b.y+b.h) ? (a.y+a.h) : (b.y+b.h);
		int h = a.h < b.h ? a.h : b.h;
		bool overlap = (double)(y1-y0)/h > 0.95;

		//Determine the center of the second panel
		int yc = b.y + b.h/2;

		//If the panels overlap, then choose based on the edge closest to
		//the side we start reading from.  Otherwise, choose based on vertical
		//position.
		return overlap ? (_l2r ? (b.x > a.x) : (b.x+b.w < a.x+a.w))
		               : (yc >= a.y+a.h);
	}
};

void Viewer::EnablePanelMode() {
	/* Find all the panels in the image, and sort them according to
	 * the given reading direction. */

	//Do the search only if it hasn't been done already.
	if (!_do_panels || (_panels.size() == 0) ||
	    (_l2r != (_readdir == DIRECTION_RIGHT))) {
		_l2r = (_readdir == DIRECTION_RIGHT);
		_panels.clear();
		_current_panel = 0;

		int masksize = _image->width * _image->height;
		mask = new unsigned char[masksize];

		if (mask) {
			//Mask out the areas between the panels.
			ComputeBackground();

			//Find all of the separate areas inside the background, and save
			//the bounds of those larger than 1/8 of the image height and width.
			for (int y = 0; y < _image->height; y++) {
				for (int x = 0; x < _image->width; x++) {
					if (mask[y*_image->width + x] == 0) {
						irect r = FloodFill(x, y, 255, true);
						if ((r.w > _image->width>>3) && (r.h > _image->height>>3)) {
							_panels.push_back(r);
						}
					}
				}
			}
		}

		if (_panels.size() > 0) {
			//Sort the panels into the correct reading order.
			sort(_panels.begin(), _panels.end(), Panel_sort());
			_do_panels = true;
		} else
			_do_panels = false;

		delete mask;
	}

	if (_do_panels) {
		/* Center display on the first panel. */
		DisplayPanel();
	}
}

void Viewer::DrawPanelRect() {
	/* Draw a rectangle around the current panel. */

	irect r = _panels[_current_panel];
	int x1 = (int)(r.x*_scale)+_imagerect.x - 1;
	int x2 = (int)((r.x+r.w)*_scale)+_imagerect.x + 1;
	int y1 = (int)(r.y*_scale)+_imagerect.y - 1;
	int y2 = (int)((r.y+r.h)*_scale)+_imagerect.y + 1;
	DrawLine(x1, y1, x2, y1, 0);
	DrawLine(x1, y2, x2, y2, 0);
	DrawLine(x1, y1, x1, y2, 0);
	DrawLine(x2, y1, x2, y2, 0);
}
