#include <unistd.h>	/* For open() lseek() write() close() unlink() */
#include <sys/mman.h>	/* For mmap() munmap() */
#include <inkview.h>	/* For Message() */
#include "ZipHandler.h"
#include "ImagesMng.h"

ZipHandler::ZipHandler(std::string& outdir) : ArchiveHandler(outdir) {
  _zip = NULL;
}

ZipHandler::~ZipHandler() {
	zip_close(_zip);
}

bool ZipHandler::IsSupportedZipImage(const char *path) {
	/* If the file is unencrypted, get the type from the magic number.
	 * Otherwise, guess based on the name. */

	struct zip_stat sb;
	struct zip_file *zf;
	char file_magic[] = { 0, 0, 0, 0, 0, 0, 0, 0 };

	if (zip_stat(_zip, path, 0, &sb) == 0) {
		if (sb.encryption_method == ZIP_EM_NONE) {
			if ((zf = zip_fopen_index(_zip, sb.index, 0))) {
				zip_fread(zf, file_magic, (sb.size < 8) ? sb.size : 8);
				zip_fclose(zf);
				return (ImagesMng::GetMagicType(file_magic) != ImagesMng::UNSUPPORTED_IMAGE);
			}
		}
	}

	return ImagesMng::IsSupportedImage(path);
}

int ZipHandler::Open(std::string& path, std::vector<std::string>& files) {
	int failed = 0, err;
	int buflen = 1024;
	char errmsg[buflen];

	if ((_zip = zip_open(path.c_str(), 0, &err))) {
		files.clear();

		zip_uint64_t numfiles = zip_get_num_entries(_zip, 0);
		for (int i = 0; i < (int)numfiles; i++) {
			const char *name = zip_get_name(_zip, i, 0);
			if (IsSupportedZipImage(name))
				files.push_back(name);
		}
	} else {
		zip_error_to_str(errmsg, buflen, err, errno);
		Message(ICON_ERROR, GetLangText("@Program_name"), errmsg, 3000);
		failed = 1;
	}

	return failed;
}

int ZipHandler::UnpackFile(std::string& name, std::string &outname) {
	int need_passwd = 0;
	struct zip_stat sb;
	struct zip_file *zf;
	int zep, sep;
	int fd;
	char *buf;

	if (zip_stat(_zip, name.c_str(), 0, &sb) == 0) {
		outname = _dir;
		size_t slash_pos = name.find_last_of("/");
		if (slash_pos != std::string::npos)
			outname += name.substr(slash_pos + 1);
		else
			outname += name;
		unlink(outname.c_str());  /* Get rid of any old copies. */

		zip_error_clear(_zip);
		if (sb.encryption_method == ZIP_EM_NONE)
			zf = zip_fopen_index(_zip, sb.index, 0);
		else if (sb.encryption_method == ZIP_EM_TRAD_PKWARE)
			zf = zip_fopen_index_encrypted(_zip, sb.index, 0, _passwd.c_str());
		else
			zf = NULL;
		if (!zf) {
			zip_error_get(_zip, &zep, &sep);
			if ((zep == ZIP_ER_WRONGPASSWD) || (zep == ZIP_ER_NOPASSWD)) {
				if ((zep = ZIP_ER_WRONGPASSWD) && _passwd.size())
					Message(ICON_ERROR, GetLangText("@Program_name"), GetLangText("@PasswordNotMatch"), 3000);
				need_passwd = 1;
			}
			outname = "";
			return need_passwd;
		}

		/* Create a sparse file and mmap it to save copying the data. */
		if ((fd = open(outname.c_str(), O_RDWR | O_CREAT, 0755)) >= 0) {
			lseek(fd, sb.size-1, SEEK_SET);
			write(fd, "a", 1);
			buf = (char*)mmap(0, sb.size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
			if (buf != MAP_FAILED) {
				zip_fread(zf, buf, sb.size);
				munmap(buf, sb.size);
			}
			zip_fclose(zf);
			close(fd);
		}
	} else
		Message(ICON_ERROR, GetLangText("@Program_name"), "UnpackZipFile", 3000);

	return need_passwd;
}
