#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fts.h>
#include <dlfcn.h>
#include <inkview.h>

/*
 * Valid answers are printed to stdout and the return code is 0.
 * Invalid answers are suppressed and the return code is non-zero.
 */

int ret = 0;
enum action_types { SHOW_MESSAGE, GET_NUMBER, GET_TEXT, GET_DIRECTORY, GET_FILENAME, GET_OPTION, QUERY, UNSUPPORTED };
enum traverse_types { RECURSIVE, FLAT };
static int traverse = RECURSIVE;
static int action = UNSUPPORTED;
static char *prog = NULL;
static char *msg = NULL;
static char *path = NULL;
static int numfiles = 0;
tocentry *toc = NULL;
#define bufsize 1024
static char buf[bufsize];
static int numexts = 0;
static char *ext[64];
static char extbuf[bufsize];
static int fd = 1;



void output(const char *fmt, ...) {
	/* This kludge is for getting around the emulator swallowing output
	 * to stdout. */
	char obuf[bufsize];
	va_list args;
	va_start(args, fmt);
	vsnprintf(obuf, bufsize, fmt, args);
	write(fd, obuf, strlen(obuf));
	va_end(args);
}

void set_active_task(void) {
	/* The latest PocketBook systems require that this task be set Active
	 * for the dialog to be shown.  We do that with the GetCurrentTask()
	 * and SetActiveTask() functions.  Not all versions of the InkView API
	 * have these functions, so we'll dynamically search for them at
	 * runtime rather than try to link to them directly. */
	void *handle;
	int (*gct)(void);
	int (*sat)(int, int);

	if ((handle = dlopen("libinkview.so", RTLD_LAZY))) {
		*(void **) (&gct) = dlsym(handle, "GetCurrentTask");
		*(void **) (&sat) = dlsym(handle, "SetActiveTask");
		if (gct && sat) {
			/* execute SetActiveTask(GetCurrentTask(), 0) */
			int task = (*gct)();
			(*sat)(task, 0);
		}
		dlclose(handle);
	}
}

void char_handler(char *s) {
	if (s && *s)
		output("%s\n", s);
	else
		ret = -1;
	CloseApp();
}

void query_handler(int button) {
	if (button == 1 || button == 2)
		output("%c\n", button == 1 ? 'y' : 'n');
	else
		ret = -1;
	CloseApp();
}

void fileselect_handler(long long pos) {
	int i;
	if (pos >= 0 && pos < numfiles) {
		if (path)
			output("%s/%s\n", path, toc[(int)pos].text);
		else
			output("%s\n", toc[(int)pos].text);
	} else
		ret = -1;

	for (i = 0; i < numfiles; i++)
		free(toc[i].text);
	free(toc);
	CloseApp();
}

int no_ext_match(char *name, short namelen) {
	int i;
	for (i = 0; i < numexts; i++)
		if (!strcasecmp(name+namelen-strlen(ext[i]), ext[i]))
			return 0;
	return 1;
}

void dlg_handler(int button) {
	CloseApp();
}

void choose_option() {
	/*
	 * We use the InkView Table of Contents dialog box as an option selector.
	 */
	int tocsize = 1024, i;
	toc = (tocentry*)malloc(tocsize);
	for (i = 0; i < numexts; i++) {
		toc[numfiles].level = 1;
		toc[numfiles].page = 0;
		toc[numfiles].position = i;
		toc[numfiles].text = strdup(ext[i]);
		numfiles++;
	}

	OpenContents(toc, numfiles, 0, fileselect_handler);
}

/* For sorting the file list alphabetically */
int compar(const void *a, const void *b) {
	return strcmp(((tocentry*)a)->text, ((tocentry*)b)->text);
}

void choose_files() {
	/*
	 * We use the InkView Table of Contents dialog box as a file selector.
	 */
	int tocsize = 1024, i;
	FTS *tree;
	FTSENT *node;
	const char *dirs[] = { path, NULL };
	int pathlen = strlen(path) + 1;	// Add one for the trailing '/'

	tree = fts_open((char * const *)dirs, FTS_PHYSICAL | FTS_NOCHDIR | FTS_NOSTAT, NULL);
	if (tree) {
		/* Walk the filesystem tree starting from the given directory.
		 * Store the names of the regular files in the toc list. */
		toc = (tocentry*)malloc(tocsize);
		while ((node = fts_read(tree))) {
			if (node->fts_info & FTS_F) {
				if ((traverse == FLAT && node->fts_level > 1) ||
				    (numexts && no_ext_match(node->fts_path, node->fts_pathlen)))
					continue;

				if ((numfiles + 1)*sizeof(tocentry) >= tocsize) {
					tocsize += 1024;
					toc = realloc(toc, tocsize);
				}

				/* Set the position after we sort the contents later. */
				toc[numfiles].level = 1;
				toc[numfiles].page = 0;
				toc[numfiles].text = strdup(node->fts_path+pathlen);  //Don't display path
				numfiles++;
			}
		}
		fts_close(tree);

		/* Sort the file list */
		qsort(toc, numfiles, sizeof(tocentry), compar);
		for (i = 0; i < numfiles; i++ ) toc[i].position = i;

		if (numfiles)
			OpenContents(toc, numfiles, 0, fileselect_handler);
		else {
			ret = -1;
			Dialog(ICON_WARNING, prog, GetLangText("@Nothing_found"), GetLangText("@Close"), NULL, dlg_handler);
		}

	} else
		ret = -1;
}

int main_handler(int type, int par1, int par2) {
	if (type == EVT_INIT) {
		set_active_task();
		switch (action) {
			case GET_NUMBER:
				buf[0] = '\0';
				OpenKeyboard(msg, buf, bufsize, KBD_NUMERIC, char_handler);
				break;
			case GET_TEXT:
				buf[0] = '\0';
				OpenKeyboard(msg, buf, bufsize, KBD_NORMAL, char_handler);
				break;
			case GET_DIRECTORY:
				buf[0] = '\0';
				OpenDirectorySelector(msg, buf, bufsize, char_handler);
				break;
			case GET_FILENAME:
				choose_files();
				break;
			case GET_OPTION:
				choose_option();
				break;
			case QUERY:
				Dialog(ICON_QUESTION, prog, msg, GetLangText("@Yes"), GetLangText("@No"), query_handler);
				break;
			case SHOW_MESSAGE:
			case UNSUPPORTED:
			default:
				Dialog(ICON_INFORMATION, prog, msg, GetLangText("@Close"), NULL, dlg_handler);
				break;
		}
	}

	//Use any key press outside the UI boxes to leave.
	if (type == EVT_KEYRELEASE || type == EVT_KEYREPEAT) {
		ret = -1;
		CloseApp();
	}

	if (type == EVT_EXIT) {
		close(fd);
	}

	return 0;
}

int main(int argc, char **argv) {
	int i;
	char *p, *delim, def_delim[] = ", ",
	     def_delim_options[] = ","; /*Allow multi-word options */
	msg = "Usage: sh_tool <OPTIONS>\n  -s <message>  Show message in a Dialog\n  -d <message>  Get directory\n  -n <message>  Get integer number input\n  -t <message>  Get text input\n  -f,-F <dir> <message> [file extension] Get filename inside directory\n  -q <message> Get response to yes/no question\n  -o <opt1,opt2,...> Get option from menu\n  -h  Show this message.\n";

	prog = argv[0];
	if (argc == 3 && !strcmp(argv[1], "-d")) {
		action = GET_DIRECTORY;
		msg = argv[2];
	} else if (argc == 3 && !strcmp(argv[1], "-s")) {
		action = SHOW_MESSAGE;
		msg = argv[2];
	} else if (argc == 3 && !strcmp(argv[1], "-n")) {
		action = GET_NUMBER;
		msg = argv[2];
	} else if (argc == 3 && !strcmp(argv[1], "-t")) {
		action = GET_TEXT;
		msg = argv[2];
	} else if (argc == 3 && !strcmp(argv[1], "-q")) {
		action = QUERY;
		msg = argv[2];
	} else if (argc >= 3 && !strcmp(argv[1], "-o")) {
		action = GET_OPTION;
		extbuf[0] = '\0';
		for (i = 2; i < argc; i++) {
			if (i != 2) /* Separate arguments with delimiter */
				strcat(extbuf, def_delim_options);
			strcat(extbuf, argv[i]);
		}
		p = extbuf;
		while ((ext[numexts] = strtok(p, def_delim_options))) {
			p = NULL;
			numexts++;
		}
	} else if (argc >= 3 && (!strcmp(argv[1], "-f") || !strcmp(argv[1], "-F"))) {
		action = GET_FILENAME;
		if (!strcmp(argv[1], "-F"))
			traverse = FLAT;
		path = argv[2];
		if (argc >= 4) {
			if (!(delim = getenv("SHIVFS")))
				delim = def_delim;
			extbuf[0] = '\0';
			for (i = 3; i < argc; i++) {
				if (i != 3)	/* Separate arguments with last delimiter in the list */
					strcat(extbuf, &delim[strlen(delim)-1]);
				strcat(extbuf, argv[i]);
			}
			p = extbuf;
			while ((ext[numexts] = strtok(p, delim))) {
				p = NULL;
				numexts++;
			}
		}

		// Strip off trailing '/' for consistent behaviour later.
		if (path[strlen(path)-1] == '/')
			path[strlen(path)-1] = '\0';
	} else
		ret = -1;

	/* dup stdout to bypass the emulator swallowing output */
	fd = dup(1);

	InkViewMain(main_handler);
	return ret;
}
